import { Injectable } from '@nestjs/common';
import mqtt from 'mqtt';
import { Observable, Subject } from 'rxjs';
import { IncomingMessage } from 'http';
import { map } from 'rxjs/operators';

/**
 * Interface representing a machine channel.
 * @interface MachineChannel
 * @property {Subject<string>} subject - The subject for the channel.
 * @property {number} refCount - The reference count for the channel.
 * @property {string} latestMessage - The latest message received on the channel.
 */
interface MachineChannel {
  subject: Subject<string>;
  refCount: number;
  latestMessage: string;
}

/**
 * Service to handle MQTT connections and subscriptions.
 * @class MqttService
 * @description This service manages MQTT connections, subscriptions, and message handling.
 */
@Injectable()
export class MqttService {
  private _client: mqtt.MqttClient;
  private _channels: Map<string, MachineChannel> = new Map();

  constructor() {
    this.mqttConnect();

    this.mqttInfoMessages();

    this.mqttGetMessageFromSubscribedTopics();
  }

  /**
   * Connect to the MQTT broker.
   */
  mqttConnect() {
    this._client = mqtt.connect(`mqtt://${process.env.MQTT_HOST_IP}`, {
      reconnectPeriod: 5000,
      connectTimeout: 30_000,
    });

    this._client.on('connect', () => {
      console.log('Connected to MQTT broker');

      // Re-subscribe to all topics after reconnect
      this._channels.forEach((_channel, topic) => {
        this._client.subscribe(topic, (err) => {
          if (err) {
            console.error(`Re-subscribe failed for ${topic}:`, err.message);
            this._channels
              .get(topic)
              ?.subject.next(`[ERROR] Resubscribe failed: ${err.message}`);
          } else {
            console.log(`Re-subscribed to topic: ${topic}`);
          }
        });
      });
    });
  }

  /**
   * Get the status of the MQTT client.
   * @return {string} Status message.
   */
  getClientStatus(): string {
    if (this._client.connected) {
      return 'MQTT client is connected';
    } else {
      return 'MQTT client is not connected';
    }
  }

  /**
   * Disconnect the MQTT client.
   */
  mqttDisconnect() {
    this._client.end(true, () => {
      console.log('MQTT client disconnected');
    });
  }

  /**
   * Get channels matching a specific topic.
   * @return {boolean} True if the pattern matches the topic, false otherwise.
   * @param pattern
   * @param topic
   */
  private mqttPatternMatches(pattern: string, topic: string): boolean {
    const patternParts = pattern.split('/');
    const topicParts = topic.split('/');

    for (let i = 0; i < patternParts.length; i++) {
      const patternPart = patternParts[i];
      const topicPart = topicParts[i];

      if (patternPart === '#') return true;
      if (!topicPart) return false;
      if (patternPart !== '+' && patternPart !== topicPart) return false;
    }

    return patternParts.length === topicParts.length;
  }

  /**
   * Get channels matching a specific topic.
   * @param messageTopic
   * @return {Map<string, MachineChannel>} Map of matching channels.
   */
  private getMatchingChannels(
    messageTopic: string,
  ): Map<string, MachineChannel> {
    const result = new Map<string, MachineChannel>();

    this._channels.forEach((entry, subscribedTopic) => {
      if (this.mqttPatternMatches(subscribedTopic, messageTopic)) {
        result.set(subscribedTopic, entry);
      }
    });

    return result;
  }

  /**
   * Get messages from subscribed topics.
   * @private
   * @description This method listens for messages on subscribed topics and updates the corresponding channels.
   */
  private mqttGetMessageFromSubscribedTopics() {
    this._client.on('message', (topic, message) => {
      const data = message.toString();
      const channels = this.getMatchingChannels(topic);

      channels.forEach((entry) => {
        console.log(data);
        entry.latestMessage = data;
        entry.subject.next(data);
      });
    });
  }

  /**
   * Handle MQTT connection errors and other events.
   * @private
   * @description This method sets up listeners for MQTT connection errors, offline status, and reconnections.
   */
  private mqttInfoMessages() {
    this._client.on('error', (error) => {
      console.error('MQTT connection error:', error.message);
      this.broadcast(`[ERROR] MQTT connection error: ${error.message}`);
    });

    this._client.on('offline', () => {
      console.warn('MQTT client is offline');
      this.broadcast('[WARN] MQTT client is offline');
    });

    this._client.on('reconnect', () => {
      console.warn('Attempting to reconnect to MQTT broker...');
      this.broadcast('[INFO] Reconnecting to broker...');
    });
  }

  /**
   * Broadcast a message to all channels.
   * @private
   * @param message
   */
  private broadcast(message: string) {
    this._channels.forEach(({ subject }) => {
      subject.next(message);
    });
  }

  /**
   * Listen to a specific machine's MQTT messages.
   * @return {Observable<string>} Observable of string messages.
   * @param topic
   */
  listenTo(topic: string): Observable<string> {
    if (!this._client.connected) {
      console.warn('MQTT client is not connected (yet)');
    }

    if (this._channels.has(topic)) {
      const existing = this._channels.get(topic);

      if (!existing) {
        throw new Error('Channel should exist but was not found');
      }

      existing.refCount++;
      return existing.subject.asObservable();
    }

    const subject = new Subject<string>();
    this._channels.set(topic, { subject, refCount: 1, latestMessage: '' });

    this._client.subscribe(topic, (err) => {
      if (err) {
        console.error(`Failed to subscribe to ${topic}:`, err.message);
        subject.next(
          `[ERROR] Subscription failed for ${topic}: ${err.message}`,
        );
      } else {
        console.log(`Subscribed to topic: ${topic}`);
      }
    });

    return subject.asObservable();
  }

  /**
   * Stop listening to a specific machine's MQTT messages.
   * @param topic
   */
  stopListeningTo(topic: string) {
    const entry = this._channels.get(topic);

    if (!entry) return;

    entry.refCount--;
    if (entry.refCount <= 0) {
      this._channels.delete(topic);
      this._client.unsubscribe(topic);
      console.log(`Unsubscribed from topic: ${topic}`);
    }
  }

  /**
   * Get the latest message for a specific machine ID.
   * @param machineId
   * @param sensor
   * @return {string} Latest message or a default message if not subscribed.
   */
  getLatestMessage(machineId: number, sensor: string): string {
    const topic = `${machineId}/${sensor}`;
    const channel = this._channels.get(topic);

    if (!channel) {
      return '[INFO] Not subscribed to this machine.';
    }

    return channel.latestMessage || '[LIVE] Awaiting new data...';
  }

  /**
   * Get live updates for a specific machine ID.
   * @param topic
   * @param request
   * @return {Observable<{ data: string }>} Observable of string messages.
   */
  getLiveUpdates(
    topic: string,
    request: IncomingMessage,
  ): Observable<{ data: string }> {
    const stream$ = this.listenTo(topic).pipe(map((data) => ({ data })));

    request.on('close', () => {
      console.log(`SSE connection closed for machine ${topic}`);
      this.stopListeningTo(`${topic}`);
    });

    return stream$;
  }

  /**
   * Get the status of the MQTT subscriptions.
   * @return {string} List of subscribed topics or a message indicating no active subscriptions.
   */
  getSubscriptionsStatus(): string {
    if (this._channels.size > 0) {
      const topics = Array.from(this._channels.entries())
        .map(([topic, channel]) => `${topic} (refCount: ${channel.refCount})`)
        .join(', ');
      return `Subscribed to ${this._channels.size} topics: ${topics}`;
    } else {
      return 'No active subscriptions';
    }
  }
}
